"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vitest_1 = require("vitest");
const utils_1 = require("./utils");
(0, vitest_1.describe)('iso', () => {
    (0, vitest_1.it)('returns ISO string for given date', () => {
        const date = new Date('2024-01-15T10:30:00.000Z');
        (0, vitest_1.expect)((0, utils_1.iso)(date)).toBe('2024-01-15T10:30:00.000Z');
    });
    (0, vitest_1.it)('returns current time ISO string when no date provided', () => {
        const before = new Date().toISOString();
        const result = (0, utils_1.iso)();
        const after = new Date().toISOString();
        (0, vitest_1.expect)(result >= before).toBe(true);
        (0, vitest_1.expect)(result <= after).toBe(true);
    });
});
(0, vitest_1.describe)('b64url', () => {
    (0, vitest_1.it)('encodes string to base64url', () => {
        const result = (0, utils_1.b64url)('hello world');
        (0, vitest_1.expect)(result).toBe(Buffer.from('hello world').toString('base64url'));
    });
    (0, vitest_1.it)('handles special characters', () => {
        const result = (0, utils_1.b64url)('test::path/to/file.ts::test-id');
        (0, vitest_1.expect)(result).toBeDefined();
        // Should not contain + or / (base64url uses - and _)
        (0, vitest_1.expect)(result).not.toMatch(/[+/]/);
    });
    (0, vitest_1.it)('produces consistent output', () => {
        const input = 'run-123::tests/foo.spec.ts::test-456';
        (0, vitest_1.expect)((0, utils_1.b64url)(input)).toBe((0, utils_1.b64url)(input));
    });
});
(0, vitest_1.describe)('inferBrowserFromProject', () => {
    vitest_1.it.each([
        ['chromium', 'chromium'],
        ['Chromium', 'chromium'],
        ['firefox', 'firefox'],
        ['Firefox', 'firefox'],
        ['webkit', 'webkit'],
        ['WebKit', 'webkit'],
        ['safari', 'safari'],
        ['Safari', 'safari'],
        ['chrome', 'chrome'],
        ['Chrome', 'chrome'],
        ['edge', 'edge'],
        ['Edge', 'edge'],
        ['electron', 'electron'],
        ['Electron Desktop', 'electron'],
    ])('infers browser from project name "%s"', (project, expected) => {
        (0, vitest_1.expect)((0, utils_1.inferBrowserFromProject)(project)).toBe(expected);
    });
    (0, vitest_1.it)('returns undefined for non-browser project names', () => {
        (0, vitest_1.expect)((0, utils_1.inferBrowserFromProject)('desktop')).toBeUndefined();
        (0, vitest_1.expect)((0, utils_1.inferBrowserFromProject)('mobile')).toBeUndefined();
        (0, vitest_1.expect)((0, utils_1.inferBrowserFromProject)('api')).toBeUndefined();
    });
    (0, vitest_1.it)('handles undefined project', () => {
        (0, vitest_1.expect)((0, utils_1.inferBrowserFromProject)(undefined)).toBeUndefined();
    });
});
(0, vitest_1.describe)('toRelativeSpec', () => {
    (0, vitest_1.it)('converts absolute path to relative', () => {
        const result = (0, utils_1.toRelativeSpec)('/home/user/project', '/home/user/project/tests/example.spec.ts');
        (0, vitest_1.expect)(result).toBe('tests/example.spec.ts');
    });
    (0, vitest_1.it)('returns relative path without leading slash', () => {
        const result = (0, utils_1.toRelativeSpec)('/home/user/project', '/home/user/project/tests/example.spec.ts');
        (0, vitest_1.expect)(result.startsWith('/')).toBe(false);
    });
    (0, vitest_1.it)('returns relative path with ../ for paths outside repo root', () => {
        const result = (0, utils_1.toRelativeSpec)('/home/user/project', '/other/path/test.ts');
        // path.relative returns a relative path with ../
        (0, vitest_1.expect)(result).toContain('..');
    });
    (0, vitest_1.it)('handles Windows-style paths', () => {
        const result = (0, utils_1.toRelativeSpec)('C:\\Users\\dev\\project', 'C:\\Users\\dev\\project\\tests\\example.spec.ts');
        (0, vitest_1.expect)(result).toBeDefined();
    });
});
(0, vitest_1.describe)('getRepoName', () => {
    const originalEnv = process.env;
    (0, vitest_1.beforeEach)(() => {
        vitest_1.vi.resetModules();
        process.env = { ...originalEnv };
        delete process.env.REPORTER_REPO_NAME;
        delete process.env.GITHUB_REPOSITORY;
    });
    (0, vitest_1.afterEach)(() => {
        process.env = originalEnv;
    });
    (0, vitest_1.it)('returns override if provided', () => {
        (0, vitest_1.expect)((0, utils_1.getRepoName)('my-custom-repo')).toBe('my-custom-repo');
    });
    (0, vitest_1.it)('extracts repo name from GITHUB_REPOSITORY', () => {
        process.env.GITHUB_REPOSITORY = 'posit-dev/positron';
        (0, vitest_1.expect)((0, utils_1.getRepoName)()).toBe('positron');
    });
    (0, vitest_1.it)('returns "local" when no repo info available', () => {
        (0, vitest_1.expect)((0, utils_1.getRepoName)()).toBe('local');
    });
    (0, vitest_1.it)('prefers REPORTER_REPO_NAME over GITHUB_REPOSITORY', () => {
        process.env.REPORTER_REPO_NAME = 'custom-name';
        process.env.GITHUB_REPOSITORY = 'posit-dev/positron';
        (0, vitest_1.expect)((0, utils_1.getRepoName)()).toBe('custom-name');
    });
});
(0, vitest_1.describe)('getRunId', () => {
    const originalEnv = process.env;
    (0, vitest_1.beforeEach)(() => {
        vitest_1.vi.resetModules();
        process.env = { ...originalEnv };
        delete process.env.GITHUB_RUN_ID;
        delete process.env.GITHUB_RUN_ATTEMPT;
    });
    (0, vitest_1.afterEach)(() => {
        process.env = originalEnv;
    });
    (0, vitest_1.it)('returns GITHUB_RUN_ID with attempt number', () => {
        process.env.GITHUB_RUN_ID = '123456789';
        process.env.GITHUB_RUN_ATTEMPT = '2';
        (0, vitest_1.expect)((0, utils_1.getRunId)()).toBe('123456789-2');
    });
    (0, vitest_1.it)('defaults attempt to 1 when not specified', () => {
        process.env.GITHUB_RUN_ID = '123456789';
        (0, vitest_1.expect)((0, utils_1.getRunId)()).toBe('123456789-1');
    });
    (0, vitest_1.it)('generates local ID when GITHUB_RUN_ID not available', () => {
        const runId = (0, utils_1.getRunId)();
        (0, vitest_1.expect)(runId).toMatch(/^local-\d+-[a-z0-9]+$/);
    });
});
(0, vitest_1.describe)('getDatePartition', () => {
    (0, vitest_1.it)('returns date in YYYY-MM-DD format', () => {
        const result = (0, utils_1.getDatePartition)();
        (0, vitest_1.expect)(result).toMatch(/^\d{4}-\d{2}-\d{2}$/);
    });
    (0, vitest_1.it)('returns current UTC date', () => {
        const now = new Date();
        const expected = now.toISOString().split('T')[0];
        (0, vitest_1.expect)((0, utils_1.getDatePartition)()).toBe(expected);
    });
});
(0, vitest_1.describe)('inferOs', () => {
    (0, vitest_1.it)('returns valid OS enum value', () => {
        const result = (0, utils_1.inferOs)();
        (0, vitest_1.expect)(['mac', 'win', 'linux']).toContain(result);
    });
});//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/utils.test.js.map