"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.writeResultsToParquet = writeResultsToParquet;
exports.writeRunEventsToParquet = writeRunEventsToParquet;
// Use @dsnp/parquetjs for Parquet file operations
// eslint-disable-next-line @typescript-eslint/no-require-imports
const parquet = require('@dsnp/parquetjs');
/**
 * Schema for test result events
 */
function getResultSchema() {
    return new parquet.ParquetSchema({
        type: { type: 'UTF8' },
        repoName: { type: 'UTF8' },
        runId: { type: 'UTF8' },
        specPath: { type: 'UTF8' },
        project: { type: 'UTF8', optional: true },
        testName: { type: 'UTF8' },
        testId: { type: 'UTF8', optional: true },
        status: { type: 'UTF8' },
        durationMs: { type: 'INT64', optional: true },
        startedAt: { type: 'UTF8', optional: true },
        attempt: { type: 'INT32', optional: true },
        retries: { type: 'INT32', optional: true },
        failureMessage: { type: 'UTF8', optional: true },
        eventId: { type: 'UTF8' },
        timestamp: { type: 'UTF8' },
        browser: { type: 'UTF8', optional: true },
        os: { type: 'UTF8', optional: true },
        branch: { type: 'UTF8', optional: true },
        sha: { type: 'UTF8', optional: true },
        // Phase 3 additions for Test Explorer
        reportUrl: { type: 'UTF8', optional: true },
        commitMessage: { type: 'UTF8', optional: true },
        commitAuthor: { type: 'UTF8', optional: true },
        titlePath: { type: 'UTF8', repeated: true, optional: true },
    });
}
/**
 * Schema for run events (started + completed)
 */
function getRunSchema() {
    return new parquet.ParquetSchema({
        type: { type: 'UTF8' },
        repoName: { type: 'UTF8' },
        runId: { type: 'UTF8' },
        // run.started fields
        startedAt: { type: 'UTF8', optional: true },
        branch: { type: 'UTF8', optional: true },
        sha: { type: 'UTF8', optional: true },
        workflowName: { type: 'UTF8', optional: true },
        jobName: { type: 'UTF8', optional: true },
        runUrl: { type: 'UTF8', optional: true },
        // Phase 3 additions for Test Explorer
        commitMessage: { type: 'UTF8', optional: true },
        commitAuthor: { type: 'UTF8', optional: true },
        // run.completed fields
        completedAt: { type: 'UTF8', optional: true },
        status: { type: 'UTF8', optional: true },
        durationMs: { type: 'INT64', optional: true },
        countsPassed: { type: 'INT32', optional: true },
        countsFailed: { type: 'INT32', optional: true },
        countsSkipped: { type: 'INT32', optional: true },
        countsFlaky: { type: 'INT32', optional: true },
    });
}
/**
 * Write test results to a Parquet file
 */
async function writeResultsToParquet(results, filePath) {
    const schema = getResultSchema();
    const writer = await parquet.ParquetWriter.openFile(schema, filePath);
    for (const result of results) {
        await writer.appendRow({
            type: result.type,
            repoName: result.repoName,
            runId: result.runId,
            specPath: result.specPath,
            project: result.project ?? null,
            testName: result.testName,
            testId: result.testId ?? null,
            status: result.status,
            durationMs: result.durationMs ?? null,
            startedAt: result.startedAt ?? null,
            attempt: result.attempt ?? null,
            retries: result.retries ?? null,
            failureMessage: result.failureMessage ?? null,
            eventId: result.eventId,
            timestamp: result.timestamp,
            browser: result.browser ?? null,
            os: result.os ?? null,
            branch: result.branch ?? null,
            sha: result.sha ?? null,
            // Phase 3 additions
            reportUrl: result.reportUrl ?? null,
            commitMessage: result.commitMessage ?? null,
            commitAuthor: result.commitAuthor ?? null,
            titlePath: result.titlePath ?? null,
        });
    }
    await writer.close();
}
/**
 * Write run events to a Parquet file
 */
async function writeRunEventsToParquet(runStarted, runCompleted, filePath) {
    const schema = getRunSchema();
    const writer = await parquet.ParquetWriter.openFile(schema, filePath);
    // Write run.started
    await writer.appendRow({
        type: runStarted.type,
        repoName: runStarted.repoName,
        runId: runStarted.runId,
        startedAt: runStarted.startedAt,
        branch: runStarted.branch ?? null,
        sha: runStarted.sha ?? null,
        workflowName: runStarted.workflowName ?? null,
        jobName: runStarted.jobName ?? null,
        runUrl: runStarted.runUrl ?? null,
        // Phase 3 additions
        commitMessage: runStarted.commitMessage ?? null,
        commitAuthor: runStarted.commitAuthor ?? null,
        completedAt: null,
        status: null,
        durationMs: null,
        countsPassed: null,
        countsFailed: null,
        countsSkipped: null,
        countsFlaky: null,
    });
    // Write run.completed if available
    if (runCompleted) {
        await writer.appendRow({
            type: runCompleted.type,
            repoName: runCompleted.repoName,
            runId: runCompleted.runId,
            startedAt: null,
            branch: null,
            sha: null,
            workflowName: null,
            jobName: null,
            runUrl: null,
            // Phase 3 additions (null for completed event)
            commitMessage: null,
            commitAuthor: null,
            completedAt: runCompleted.completedAt,
            status: runCompleted.status,
            durationMs: runCompleted.durationMs ?? null,
            countsPassed: runCompleted.counts.passed,
            countsFailed: runCompleted.counts.failed,
            countsSkipped: runCompleted.counts.skipped,
            countsFlaky: runCompleted.counts.flaky,
        });
    }
    await writer.close();
}//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/storage/parquet.js.map