"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vitest_1 = require("vitest");
const api_1 = require("./api");
// Mock global fetch
const mockFetch = vitest_1.vi.fn();
global.fetch = mockFetch;
(0, vitest_1.describe)('createApiWriter', () => {
    (0, vitest_1.beforeEach)(() => {
        mockFetch.mockReset();
        vitest_1.vi.useFakeTimers();
    });
    (0, vitest_1.afterEach)(() => {
        vitest_1.vi.useRealTimers();
    });
    const config = { url: 'http://localhost:8000' };
    const mockResultEvent = {
        type: 'result.created',
        repoName: 'test-repo',
        runId: 'run-123',
        specPath: 'tests/example.spec.ts',
        testName: 'should work',
        status: 'passed',
        timestamp: '2024-01-15T10:30:00.000Z',
        eventId: 'abc123',
    };
    const mockRunStartedEvent = {
        type: 'run.started',
        repoName: 'test-repo',
        runId: 'run-123',
        startedAt: '2024-01-15T10:30:00.000Z',
    };
    const mockRunCompletedEvent = {
        type: 'run.completed',
        repoName: 'test-repo',
        runId: 'run-123',
        completedAt: '2024-01-15T10:35:00.000Z',
        status: 'passed',
        counts: { passed: 10, failed: 0, skipped: 0, flaky: 0 },
    };
    (0, vitest_1.describe)('writeResults', () => {
        (0, vitest_1.it)('sends POST request with correct URL and body', async () => {
            mockFetch.mockResolvedValueOnce({
                ok: true,
                json: async () => ({ status: 'success', path: 'test-results/...' }),
            });
            const writer = (0, api_1.createApiWriter)(config);
            const success = await writer.writeResults([mockResultEvent], 'test-repo', '2024-01-15', 'run-123', '001');
            (0, vitest_1.expect)(success).toBe(true);
            (0, vitest_1.expect)(mockFetch).toHaveBeenCalledTimes(1);
            const [url, options] = mockFetch.mock.calls[0];
            (0, vitest_1.expect)(url).toContain('http://localhost:8000/test-results');
            (0, vitest_1.expect)(url).toContain('repo=test-repo');
            (0, vitest_1.expect)(url).toContain('run_id=run-123');
            (0, vitest_1.expect)(url).toContain('date=2024-01-15');
            (0, vitest_1.expect)(url).toContain('batch_id=001');
            (0, vitest_1.expect)(options.method).toBe('POST');
            (0, vitest_1.expect)(options.headers['Content-Type']).toBe('application/json');
            (0, vitest_1.expect)(JSON.parse(options.body)).toEqual([mockResultEvent]);
        });
        (0, vitest_1.it)('returns true for empty results array', async () => {
            const writer = (0, api_1.createApiWriter)(config);
            const success = await writer.writeResults([], 'test-repo', '2024-01-15', 'run-123');
            (0, vitest_1.expect)(success).toBe(true);
            (0, vitest_1.expect)(mockFetch).not.toHaveBeenCalled();
        });
        (0, vitest_1.it)('retries on failure and succeeds', async () => {
            mockFetch
                .mockRejectedValueOnce(new Error('Network error'))
                .mockResolvedValueOnce({
                ok: true,
                json: async () => ({ status: 'success' }),
            });
            const writer = (0, api_1.createApiWriter)(config);
            const resultPromise = writer.writeResults([mockResultEvent], 'test-repo', '2024-01-15', 'run-123');
            // Fast-forward through retry delay
            await vitest_1.vi.advanceTimersByTimeAsync(1000);
            const success = await resultPromise;
            (0, vitest_1.expect)(success).toBe(true);
            (0, vitest_1.expect)(mockFetch).toHaveBeenCalledTimes(2);
        });
        (0, vitest_1.it)('returns false after max retries', async () => {
            mockFetch.mockRejectedValue(new Error('Network error'));
            const writer = (0, api_1.createApiWriter)(config);
            const resultPromise = writer.writeResults([mockResultEvent], 'test-repo', '2024-01-15', 'run-123');
            // Fast-forward through all retry delays (1s + 2s + 4s)
            await vitest_1.vi.advanceTimersByTimeAsync(1000);
            await vitest_1.vi.advanceTimersByTimeAsync(2000);
            await vitest_1.vi.advanceTimersByTimeAsync(4000);
            const success = await resultPromise;
            (0, vitest_1.expect)(success).toBe(false);
            (0, vitest_1.expect)(mockFetch).toHaveBeenCalledTimes(3);
        });
        (0, vitest_1.it)('returns false on non-success response', async () => {
            mockFetch.mockResolvedValue({
                ok: true,
                json: async () => ({ status: 'error', message: 'Validation failed' }),
            });
            const writer = (0, api_1.createApiWriter)(config);
            const success = await writer.writeResults([mockResultEvent], 'test-repo', '2024-01-15', 'run-123');
            (0, vitest_1.expect)(success).toBe(false);
        });
        (0, vitest_1.it)('handles HTTP error responses', async () => {
            mockFetch.mockResolvedValue({
                ok: false,
                status: 400,
                text: async () => 'Bad Request',
            });
            const writer = (0, api_1.createApiWriter)(config);
            const resultPromise = writer.writeResults([mockResultEvent], 'test-repo', '2024-01-15', 'run-123');
            // Fast-forward through retries
            await vitest_1.vi.advanceTimersByTimeAsync(7000);
            const success = await resultPromise;
            (0, vitest_1.expect)(success).toBe(false);
        });
    });
    (0, vitest_1.describe)('writeRunEvent', () => {
        (0, vitest_1.it)('sends run.started event correctly', async () => {
            mockFetch.mockResolvedValueOnce({
                ok: true,
                json: async () => ({ status: 'success', path: 'test-results/...' }),
            });
            const writer = (0, api_1.createApiWriter)(config);
            const success = await writer.writeRunEvent(mockRunStartedEvent, 'test-repo', '2024-01-15', 'run-123');
            (0, vitest_1.expect)(success).toBe(true);
            (0, vitest_1.expect)(mockFetch).toHaveBeenCalledTimes(1);
            const [url, options] = mockFetch.mock.calls[0];
            (0, vitest_1.expect)(url).toContain('http://localhost:8000/test-results/run');
            (0, vitest_1.expect)(url).toContain('repo=test-repo');
            (0, vitest_1.expect)(JSON.parse(options.body)).toEqual(mockRunStartedEvent);
        });
        (0, vitest_1.it)('sends run.completed event correctly', async () => {
            mockFetch.mockResolvedValueOnce({
                ok: true,
                json: async () => ({ status: 'success' }),
            });
            const writer = (0, api_1.createApiWriter)(config);
            const success = await writer.writeRunEvent(mockRunCompletedEvent, 'test-repo', '2024-01-15', 'run-123');
            (0, vitest_1.expect)(success).toBe(true);
            const [, options] = mockFetch.mock.calls[0];
            (0, vitest_1.expect)(JSON.parse(options.body)).toEqual(mockRunCompletedEvent);
        });
        (0, vitest_1.it)('retries on failure', async () => {
            mockFetch
                .mockRejectedValueOnce(new Error('Network error'))
                .mockResolvedValueOnce({
                ok: true,
                json: async () => ({ status: 'success' }),
            });
            const writer = (0, api_1.createApiWriter)(config);
            const resultPromise = writer.writeRunEvent(mockRunStartedEvent, 'test-repo', '2024-01-15', 'run-123');
            await vitest_1.vi.advanceTimersByTimeAsync(1000);
            const success = await resultPromise;
            (0, vitest_1.expect)(success).toBe(true);
            (0, vitest_1.expect)(mockFetch).toHaveBeenCalledTimes(2);
        });
    });
    (0, vitest_1.describe)('verbose logging', () => {
        (0, vitest_1.it)('logs when verbose is true', async () => {
            const consoleSpy = vitest_1.vi.spyOn(console, 'log').mockImplementation(() => { });
            mockFetch.mockResolvedValueOnce({
                ok: true,
                json: async () => ({ status: 'success', path: 'test-path' }),
            });
            const writer = (0, api_1.createApiWriter)(config, true);
            await writer.writeResults([mockResultEvent], 'test-repo', '2024-01-15', 'run-123');
            (0, vitest_1.expect)(consoleSpy).toHaveBeenCalled();
            consoleSpy.mockRestore();
        });
        (0, vitest_1.it)('does not log when verbose is false', async () => {
            const consoleSpy = vitest_1.vi.spyOn(console, 'log').mockImplementation(() => { });
            mockFetch.mockResolvedValueOnce({
                ok: true,
                json: async () => ({ status: 'success' }),
            });
            const writer = (0, api_1.createApiWriter)(config, false);
            await writer.writeResults([mockResultEvent], 'test-repo', '2024-01-15', 'run-123');
            // Should only be called by internal logging, not verbose
            (0, vitest_1.expect)(consoleSpy).not.toHaveBeenCalledWith(vitest_1.expect.stringContaining('[ParquetReporter]'), vitest_1.expect.anything());
            consoleSpy.mockRestore();
        });
    });
});//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/storage/api.test.js.map