"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createApiWriter = createApiWriter;
/**
 * Sleep for a given number of milliseconds
 */
function sleep(ms) {
    return new Promise((resolve) => setTimeout(resolve, ms));
}
/**
 * POST JSON to API with retry logic
 * @returns Response JSON on success, null on failure (after retries)
 */
async function postWithRetry(url, body, params, maxRetries = 3, apiKey) {
    // Build URL with query params
    const queryString = new URLSearchParams(Object.entries(params).filter(([, v]) => v !== undefined && v !== '')).toString();
    const fullUrl = queryString ? `${url}?${queryString}` : url;
    // Build headers with optional API key auth
    const headers = { 'Content-Type': 'application/json' };
    if (apiKey) {
        headers['Authorization'] = `Key ${apiKey}`;
    }
    for (let attempt = 1; attempt <= maxRetries; attempt++) {
        try {
            const response = await fetch(fullUrl, {
                method: 'POST',
                headers,
                body: JSON.stringify(body),
            });
            if (!response.ok) {
                const errorText = await response.text().catch(() => 'Unknown error');
                throw new Error(`HTTP ${response.status}: ${errorText}`);
            }
            return await response.json();
        }
        catch (err) {
            const errorMessage = err instanceof Error ? err.message : String(err);
            if (attempt === maxRetries) {
                console.warn(`  ⚠ API failed after ${maxRetries} attempts: ${errorMessage}`);
                return null; // Warn but don't throw - tests continue
            }
            // Exponential backoff: 1s, 2s, 4s
            const delayMs = 1000 * Math.pow(2, attempt - 1);
            await sleep(delayMs);
        }
    }
    return null;
}
/**
 * Create an API writer for sending test results to Plumber API
 */
function createApiWriter(config, verbose = false) {
    const log = (message) => {
        if (verbose) {
            console.log(message);
        }
    };
    return {
        async writeResults(results, repo, date, runId, batchId) {
            if (results.length === 0) {
                return true;
            }
            log(`  ↑ sending ${results.length} results to ${config.url}/test-results`);
            const response = await postWithRetry(`${config.url}/test-results`, results, {
                repo,
                run_id: runId,
                date,
                batch_id: batchId || '',
            }, 3, config.apiKey);
            const success = response !== null && response.status === 'success';
            if (success) {
                log(`  ✓ results saved`);
            }
            return success;
        },
        async writeRunEvent(event, repo, date, runId) {
            log(`  ↑ sending ${event.type} event to ${config.url}/test-results/run`);
            const response = await postWithRetry(`${config.url}/test-results/run`, event, {
                repo,
                run_id: runId,
                date,
            }, 3, config.apiKey);
            const success = response !== null && response.status === 'success';
            if (success) {
                log(`  ✓ ${event.type} event saved`);
            }
            return success;
        },
        async writeData(data, repo, date, runId) {
            if (data.length === 0) {
                return true;
            }
            log(`  ↑ sending ${data.length} consolidated results to ${config.url}/test-results/data`);
            const response = await postWithRetry(`${config.url}/test-results/data`, data, {
                repo,
                run_id: runId,
                date,
            }, 3, config.apiKey);
            const success = response !== null && response.status === 'success';
            if (success) {
                log(`  ✓ consolidated data saved`);
            }
            return success;
        },
    };
}//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/storage/api.js.map