"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vitest_1 = require("vitest");
const schemas_1 = require("./schemas");
(0, vitest_1.describe)('ResultCreatedEventSchema', () => {
    const validEvent = {
        type: 'result.created',
        repoName: 'test-repo',
        runId: 'run-123',
        specPath: 'tests/example.spec.ts',
        testName: 'should work',
        status: 'passed',
        timestamp: '2024-01-15T10:30:00.000Z',
        eventId: 'abc123',
    };
    (0, vitest_1.it)('accepts valid minimal event', () => {
        const result = schemas_1.ResultCreatedEventSchema.safeParse(validEvent);
        (0, vitest_1.expect)(result.success).toBe(true);
    });
    (0, vitest_1.it)('accepts valid event with all optional fields', () => {
        const fullEvent = {
            ...validEvent,
            project: 'chromium',
            testId: 'test-id-123',
            durationMs: 1500,
            startedAt: '2024-01-15T10:30:00.000Z',
            attempt: 1,
            retries: 0,
            failureMessage: undefined,
            browser: 'chromium',
            os: 'mac',
            branch: 'main',
            sha: 'abc123def',
            reportUrl: 'https://example.com/report',
            commitMessage: 'fix: something',
            commitAuthor: 'dev@example.com',
            titlePath: ['Suite', 'Nested', 'Test'],
        };
        const result = schemas_1.ResultCreatedEventSchema.safeParse(fullEvent);
        (0, vitest_1.expect)(result.success).toBe(true);
    });
    (0, vitest_1.describe)('required fields', () => {
        (0, vitest_1.it)('rejects missing type', () => {
            const { type, ...eventWithoutType } = validEvent;
            const result = schemas_1.ResultCreatedEventSchema.safeParse(eventWithoutType);
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects missing repoName', () => {
            const { repoName, ...event } = validEvent;
            const result = schemas_1.ResultCreatedEventSchema.safeParse(event);
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects missing runId', () => {
            const { runId, ...event } = validEvent;
            const result = schemas_1.ResultCreatedEventSchema.safeParse(event);
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects missing specPath', () => {
            const { specPath, ...event } = validEvent;
            const result = schemas_1.ResultCreatedEventSchema.safeParse(event);
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects missing testName', () => {
            const { testName, ...event } = validEvent;
            const result = schemas_1.ResultCreatedEventSchema.safeParse(event);
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects missing status', () => {
            const { status, ...event } = validEvent;
            const result = schemas_1.ResultCreatedEventSchema.safeParse(event);
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects missing timestamp', () => {
            const { timestamp, ...event } = validEvent;
            const result = schemas_1.ResultCreatedEventSchema.safeParse(event);
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects missing eventId', () => {
            const { eventId, ...event } = validEvent;
            const result = schemas_1.ResultCreatedEventSchema.safeParse(event);
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects empty string for required fields', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                repoName: '',
            });
            (0, vitest_1.expect)(result.success).toBe(false);
        });
    });
    (0, vitest_1.describe)('status enum', () => {
        vitest_1.it.each(['passed', 'failed', 'skipped', 'flaky'])('accepts status "%s"', (status) => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({ ...validEvent, status });
            (0, vitest_1.expect)(result.success).toBe(true);
        });
        (0, vitest_1.it)('rejects invalid status', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                status: 'invalid',
            });
            (0, vitest_1.expect)(result.success).toBe(false);
        });
    });
    (0, vitest_1.describe)('os enum', () => {
        vitest_1.it.each(['mac', 'win', 'linux'])('accepts os "%s"', (os) => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({ ...validEvent, os });
            (0, vitest_1.expect)(result.success).toBe(true);
        });
        (0, vitest_1.it)('rejects invalid os', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                os: 'windows',
            });
            (0, vitest_1.expect)(result.success).toBe(false);
        });
    });
    (0, vitest_1.describe)('numeric fields', () => {
        (0, vitest_1.it)('rejects negative durationMs', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                durationMs: -1,
            });
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects non-positive attempt', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                attempt: 0,
            });
            (0, vitest_1.expect)(result.success).toBe(false);
        });
        (0, vitest_1.it)('rejects negative retries', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                retries: -1,
            });
            (0, vitest_1.expect)(result.success).toBe(false);
        });
    });
    (0, vitest_1.describe)('timestamp format', () => {
        (0, vitest_1.it)('accepts valid ISO 8601 timestamp', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                timestamp: '2024-01-15T10:30:00.000Z',
            });
            (0, vitest_1.expect)(result.success).toBe(true);
        });
        (0, vitest_1.it)('rejects invalid timestamp format', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                timestamp: '2024-01-15',
            });
            (0, vitest_1.expect)(result.success).toBe(false);
        });
    });
    (0, vitest_1.describe)('URL validation', () => {
        (0, vitest_1.it)('accepts valid URL for reportUrl', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                reportUrl: 'https://example.com/report/123',
            });
            (0, vitest_1.expect)(result.success).toBe(true);
        });
        (0, vitest_1.it)('rejects invalid URL for reportUrl', () => {
            const result = schemas_1.ResultCreatedEventSchema.safeParse({
                ...validEvent,
                reportUrl: 'not-a-url',
            });
            (0, vitest_1.expect)(result.success).toBe(false);
        });
    });
});
(0, vitest_1.describe)('RunStartedEventSchema', () => {
    const validEvent = {
        type: 'run.started',
        repoName: 'test-repo',
        runId: 'run-123',
        startedAt: '2024-01-15T10:30:00.000Z',
    };
    (0, vitest_1.it)('accepts valid minimal event', () => {
        const result = schemas_1.RunStartedEventSchema.safeParse(validEvent);
        (0, vitest_1.expect)(result.success).toBe(true);
    });
    (0, vitest_1.it)('accepts valid event with all optional fields', () => {
        const fullEvent = {
            ...validEvent,
            branch: 'main',
            sha: 'abc123',
            workflowName: 'CI',
            jobName: 'test',
            runUrl: 'https://github.com/org/repo/actions/runs/123',
            commitMessage: 'fix: something',
            commitAuthor: 'dev@example.com',
        };
        const result = schemas_1.RunStartedEventSchema.safeParse(fullEvent);
        (0, vitest_1.expect)(result.success).toBe(true);
    });
    (0, vitest_1.it)('rejects missing startedAt', () => {
        const { startedAt, ...event } = validEvent;
        const result = schemas_1.RunStartedEventSchema.safeParse(event);
        (0, vitest_1.expect)(result.success).toBe(false);
    });
    (0, vitest_1.it)('rejects wrong type literal', () => {
        const result = schemas_1.RunStartedEventSchema.safeParse({
            ...validEvent,
            type: 'run.completed',
        });
        (0, vitest_1.expect)(result.success).toBe(false);
    });
});
(0, vitest_1.describe)('RunCompletedEventSchema', () => {
    const validEvent = {
        type: 'run.completed',
        repoName: 'test-repo',
        runId: 'run-123',
        completedAt: '2024-01-15T10:35:00.000Z',
        status: 'passed',
        counts: {
            passed: 10,
            failed: 0,
            skipped: 2,
            flaky: 1,
        },
    };
    (0, vitest_1.it)('accepts valid event', () => {
        const result = schemas_1.RunCompletedEventSchema.safeParse(validEvent);
        (0, vitest_1.expect)(result.success).toBe(true);
    });
    (0, vitest_1.it)('accepts valid event with optional durationMs', () => {
        const result = schemas_1.RunCompletedEventSchema.safeParse({
            ...validEvent,
            durationMs: 60000,
        });
        (0, vitest_1.expect)(result.success).toBe(true);
    });
    (0, vitest_1.it)('rejects missing completedAt', () => {
        const { completedAt, ...event } = validEvent;
        const result = schemas_1.RunCompletedEventSchema.safeParse(event);
        (0, vitest_1.expect)(result.success).toBe(false);
    });
    (0, vitest_1.it)('rejects missing status', () => {
        const { status, ...event } = validEvent;
        const result = schemas_1.RunCompletedEventSchema.safeParse(event);
        (0, vitest_1.expect)(result.success).toBe(false);
    });
    (0, vitest_1.it)('rejects missing counts', () => {
        const { counts, ...event } = validEvent;
        const result = schemas_1.RunCompletedEventSchema.safeParse(event);
        (0, vitest_1.expect)(result.success).toBe(false);
    });
    (0, vitest_1.it)('rejects incomplete counts object', () => {
        const result = schemas_1.RunCompletedEventSchema.safeParse({
            ...validEvent,
            counts: { passed: 10, failed: 0 }, // missing skipped and flaky
        });
        (0, vitest_1.expect)(result.success).toBe(false);
    });
    (0, vitest_1.it)('rejects negative count values', () => {
        const result = schemas_1.RunCompletedEventSchema.safeParse({
            ...validEvent,
            counts: { passed: -1, failed: 0, skipped: 0, flaky: 0 },
        });
        (0, vitest_1.expect)(result.success).toBe(false);
    });
});
(0, vitest_1.describe)('validation helpers', () => {
    (0, vitest_1.describe)('validateResultEvent', () => {
        (0, vitest_1.it)('returns validated data for valid event', () => {
            const event = {
                type: 'result.created',
                repoName: 'test-repo',
                runId: 'run-123',
                specPath: 'tests/example.spec.ts',
                testName: 'should work',
                status: 'passed',
                timestamp: '2024-01-15T10:30:00.000Z',
                eventId: 'abc123',
            };
            const result = (0, schemas_1.validateResultEvent)(event);
            (0, vitest_1.expect)(result).toEqual(event);
        });
        (0, vitest_1.it)('throws for invalid event', () => {
            (0, vitest_1.expect)(() => (0, schemas_1.validateResultEvent)({ type: 'result.created' })).toThrow();
        });
    });
    (0, vitest_1.describe)('safeValidateResultEvent', () => {
        (0, vitest_1.it)('returns success true for valid event', () => {
            const event = {
                type: 'result.created',
                repoName: 'test-repo',
                runId: 'run-123',
                specPath: 'tests/example.spec.ts',
                testName: 'should work',
                status: 'passed',
                timestamp: '2024-01-15T10:30:00.000Z',
                eventId: 'abc123',
            };
            const result = (0, schemas_1.safeValidateResultEvent)(event);
            (0, vitest_1.expect)(result.success).toBe(true);
            if (result.success) {
                (0, vitest_1.expect)(result.data).toEqual(event);
            }
        });
        (0, vitest_1.it)('returns success false for invalid event', () => {
            const result = (0, schemas_1.safeValidateResultEvent)({ type: 'result.created' });
            (0, vitest_1.expect)(result.success).toBe(false);
            if (!result.success) {
                (0, vitest_1.expect)(result.error).toBeDefined();
            }
        });
    });
});//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/schemas.test.js.map