"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConsolidatedDataEventSchema = exports.OutboundEventSchema = exports.RunCompletedEventSchema = exports.CountsSchema = exports.RunStartedEventSchema = exports.ResultCreatedEventSchema = exports.ResultStatusSchema = exports.SCHEMA_VERSION = void 0;
exports.validateConsolidatedDataEvent = validateConsolidatedDataEvent;
exports.safeValidateConsolidatedDataEvent = safeValidateConsolidatedDataEvent;
exports.validateResultEvent = validateResultEvent;
exports.validateRunStartedEvent = validateRunStartedEvent;
exports.validateRunCompletedEvent = validateRunCompletedEvent;
exports.safeValidateResultEvent = safeValidateResultEvent;
exports.safeValidateRunStartedEvent = safeValidateRunStartedEvent;
exports.safeValidateRunCompletedEvent = safeValidateRunCompletedEvent;
const zod_1 = require("zod");
/**
 * Schema version for tracking breaking changes
 * Increment when making incompatible schema changes
 */
exports.SCHEMA_VERSION = '1.0.0';
/**
 * Result status enum
 */
exports.ResultStatusSchema = zod_1.z.enum(['passed', 'failed', 'skipped', 'flaky']);
/**
 * Test result event schema
 * Emitted for each test completion (including retries)
 */
exports.ResultCreatedEventSchema = zod_1.z.object({
    type: zod_1.z.literal('result.created'),
    repoName: zod_1.z.string().min(1),
    runId: zod_1.z.string().min(1),
    specPath: zod_1.z.string().min(1),
    project: zod_1.z.string().optional(),
    testName: zod_1.z.string().min(1),
    testId: zod_1.z.string().optional(),
    status: exports.ResultStatusSchema,
    durationMs: zod_1.z.number().int().nonnegative().optional(),
    startedAt: zod_1.z.string().datetime().optional(),
    attempt: zod_1.z.number().int().positive().optional(),
    retries: zod_1.z.number().int().nonnegative().optional(),
    failureMessage: zod_1.z.string().optional(),
    eventId: zod_1.z.string().min(1),
    timestamp: zod_1.z.string().datetime(),
    browser: zod_1.z.string().optional(),
    os: zod_1.z.enum(['mac', 'win', 'linux', 'ubuntu', 'debian', 'rhel', 'suse']).optional(),
    branch: zod_1.z.string().optional(),
    sha: zod_1.z.string().optional(),
    // Phase 3 additions for Test Explorer
    reportUrl: zod_1.z.string().url().optional(),
    commitMessage: zod_1.z.string().optional(),
    commitAuthor: zod_1.z.string().optional(),
    titlePath: zod_1.z.array(zod_1.z.string()).optional(),
});
/**
 * Run started event schema
 * Emitted once at the beginning of a test run
 */
exports.RunStartedEventSchema = zod_1.z.object({
    type: zod_1.z.literal('run.started'),
    repoName: zod_1.z.string().min(1),
    runId: zod_1.z.string().min(1),
    startedAt: zod_1.z.string().datetime(),
    branch: zod_1.z.string().optional(),
    sha: zod_1.z.string().optional(),
    workflowName: zod_1.z.string().optional(),
    jobName: zod_1.z.string().optional(),
    runUrl: zod_1.z.string().url().optional(),
    // Phase 3 additions
    commitMessage: zod_1.z.string().optional(),
    commitAuthor: zod_1.z.string().optional(),
    // Workflow trigger order (GITHUB_RUN_NUMBER) - sequential per workflow, for sorting
    runNumber: zod_1.z.number().int().positive().optional(),
    // Raw GITHUB_RUN_ID for building GitHub Actions URLs (runId field includes attempt/shard suffixes)
    githubRunId: zod_1.z.string().optional(),
});
/**
 * Counts object for run completion
 */
exports.CountsSchema = zod_1.z.object({
    passed: zod_1.z.number().int().nonnegative(),
    failed: zod_1.z.number().int().nonnegative(),
    skipped: zod_1.z.number().int().nonnegative(),
    flaky: zod_1.z.number().int().nonnegative(),
});
/**
 * Run completed event schema
 * Emitted once at the end of a test run
 */
exports.RunCompletedEventSchema = zod_1.z.object({
    type: zod_1.z.literal('run.completed'),
    repoName: zod_1.z.string().min(1),
    runId: zod_1.z.string().min(1),
    completedAt: zod_1.z.string().datetime(),
    status: zod_1.z.string(),
    durationMs: zod_1.z.number().int().nonnegative().optional(),
    counts: exports.CountsSchema,
});
/**
 * Union of all outbound event types
 */
exports.OutboundEventSchema = zod_1.z.discriminatedUnion('type', [
    exports.ResultCreatedEventSchema,
    exports.RunStartedEventSchema,
    exports.RunCompletedEventSchema,
]);
/**
 * Consolidated data event schema
 * Combines test result with run metadata for single-file storage
 * Each row represents a test result with embedded run context
 */
exports.ConsolidatedDataEventSchema = zod_1.z.object({
    // Test result fields
    repoName: zod_1.z.string().min(1),
    runId: zod_1.z.string().min(1),
    specPath: zod_1.z.string().min(1),
    project: zod_1.z.string().optional(),
    testName: zod_1.z.string().min(1),
    testId: zod_1.z.string().optional(),
    status: exports.ResultStatusSchema,
    durationMs: zod_1.z.number().int().nonnegative().optional(),
    startedAt: zod_1.z.string().datetime().optional(),
    attempt: zod_1.z.number().int().positive().optional(),
    retries: zod_1.z.number().int().nonnegative().optional(),
    failureMessage: zod_1.z.string().optional(),
    eventId: zod_1.z.string().min(1),
    timestamp: zod_1.z.string().datetime(),
    browser: zod_1.z.string().optional(),
    os: zod_1.z.enum(['mac', 'win', 'linux', 'ubuntu', 'debian', 'rhel', 'suse']).optional(),
    branch: zod_1.z.string().optional(),
    sha: zod_1.z.string().optional(),
    reportUrl: zod_1.z.string().url().optional(),
    commitMessage: zod_1.z.string().optional(),
    commitAuthor: zod_1.z.string().optional(),
    titlePath: zod_1.z.array(zod_1.z.string()).optional(),
    // Run metadata fields (embedded in each result)
    runStartedAt: zod_1.z.string().datetime(),
    runCompletedAt: zod_1.z.string().datetime().optional(),
    runStatus: zod_1.z.string().optional(),
    runDurationMs: zod_1.z.number().int().nonnegative().optional(),
    workflowName: zod_1.z.string().optional(),
    jobName: zod_1.z.string().optional(),
    runUrl: zod_1.z.string().url().optional(),
    runNumber: zod_1.z.number().int().positive().optional(),
    githubRunId: zod_1.z.string().optional(),
    // Counts (populated at run completion)
    countsPassed: zod_1.z.number().int().nonnegative().optional(),
    countsFailed: zod_1.z.number().int().nonnegative().optional(),
    countsSkipped: zod_1.z.number().int().nonnegative().optional(),
    countsFlaky: zod_1.z.number().int().nonnegative().optional(),
});
/**
 * Validate a consolidated data event, throwing if invalid
 */
function validateConsolidatedDataEvent(event) {
    return exports.ConsolidatedDataEventSchema.parse(event);
}
/**
 * Safe validation for consolidated data event
 */
function safeValidateConsolidatedDataEvent(event) {
    return exports.ConsolidatedDataEventSchema.safeParse(event);
}
/**
 * Validate a result event, throwing if invalid
 */
function validateResultEvent(event) {
    return exports.ResultCreatedEventSchema.parse(event);
}
/**
 * Validate a run started event, throwing if invalid
 */
function validateRunStartedEvent(event) {
    return exports.RunStartedEventSchema.parse(event);
}
/**
 * Validate a run completed event, throwing if invalid
 */
function validateRunCompletedEvent(event) {
    return exports.RunCompletedEventSchema.parse(event);
}
/**
 * Safe validation that returns result instead of throwing
 */
function safeValidateResultEvent(event) {
    return exports.ResultCreatedEventSchema.safeParse(event);
}
function safeValidateRunStartedEvent(event) {
    return exports.RunStartedEventSchema.safeParse(event);
}
function safeValidateRunCompletedEvent(event) {
    return exports.RunCompletedEventSchema.safeParse(event);
}//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/schemas.js.map