"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const vitest_1 = require("vitest");
// Mock the API writer module
vitest_1.vi.mock('./storage/api', () => ({
    createApiWriter: vitest_1.vi.fn(() => ({
        writeResults: vitest_1.vi.fn().mockResolvedValue(true),
        writeRunEvent: vitest_1.vi.fn().mockResolvedValue(true),
    })),
}));
// Mock utils to control IDs
vitest_1.vi.mock('./utils', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        getRunId: () => 'test-run-123',
        getDatePartition: () => '2024-01-15',
        getRepoName: (override) => override || 'test-repo',
    };
});
const reporter_1 = __importDefault(require("./reporter"));
const api_1 = require("./storage/api");
(0, vitest_1.describe)('ParquetReporter', () => {
    let mockApiWriter;
    (0, vitest_1.beforeEach)(() => {
        vitest_1.vi.clearAllMocks();
        mockApiWriter = {
            writeResults: vitest_1.vi.fn().mockResolvedValue(true),
            writeRunEvent: vitest_1.vi.fn().mockResolvedValue(true),
        };
        vitest_1.vi.mocked(api_1.createApiWriter).mockReturnValue(mockApiWriter);
    });
    const createMockConfig = () => ({
        projects: [{ name: 'chromium', retries: 1 }],
    });
    const createMockSuite = (tests) => ({
        suites: [],
        tests: tests.map((t) => ({
            type: 'test',
            location: { file: t.file },
            title: t.title,
        })),
    });
    const createMockTest = (id, title, file) => ({
        id,
        title,
        location: { file, line: 1, column: 1 },
        titlePath: () => ['chromium', 'Suite', title],
    });
    const createMockResult = (status, duration, retry = 0, error) => ({
        status,
        duration,
        retry,
        startTime: new Date(),
        error,
    });
    (0, vitest_1.describe)('initialization', () => {
        (0, vitest_1.it)('initializes in dev mode', () => {
            const reporter = new reporter_1.default({ mode: 'dev' });
            (0, vitest_1.expect)(api_1.createApiWriter).toHaveBeenCalledWith({ url: 'http://localhost:8000' }, false);
        });
        (0, vitest_1.it)('initializes in prod mode', () => {
            const reporter = new reporter_1.default({ mode: 'prod' });
            (0, vitest_1.expect)(api_1.createApiWriter).toHaveBeenCalledWith({ url: 'https://connect.posit.it/e2e-test-insights-api' }, false);
        });
        (0, vitest_1.it)('initializes with verbose logging', () => {
            const reporter = new reporter_1.default({ mode: 'dev', verbose: true });
            (0, vitest_1.expect)(api_1.createApiWriter).toHaveBeenCalledWith({ url: 'http://localhost:8000' }, true);
        });
        (0, vitest_1.it)('disables when mode is disabled', async () => {
            const reporter = new reporter_1.default({ mode: 'disabled' });
            await reporter.onBegin(createMockConfig(), createMockSuite([]));
            await reporter.onEnd({ status: 'passed' });
            (0, vitest_1.expect)(mockApiWriter.writeRunEvent).not.toHaveBeenCalled();
            (0, vitest_1.expect)(mockApiWriter.writeResults).not.toHaveBeenCalled();
        });
    });
    (0, vitest_1.describe)('complete test run - all passing', () => {
        (0, vitest_1.it)('sends run.started, results, and run.completed events', async () => {
            const reporter = new reporter_1.default({ mode: 'dev', flushThreshold: 100 });
            const config = createMockConfig();
            const suite = createMockSuite([
                { file: '/project/tests/a.spec.ts', title: 'test a' },
                { file: '/project/tests/b.spec.ts', title: 'test b' },
            ]);
            // Begin run
            await reporter.onBegin(config, suite);
            // Verify run.started was sent
            (0, vitest_1.expect)(mockApiWriter.writeRunEvent).toHaveBeenCalledTimes(1);
            (0, vitest_1.expect)(mockApiWriter.writeRunEvent.mock.calls[0][0]).toMatchObject({
                type: 'run.started',
                repoName: 'test-repo',
                runId: 'test-run-123',
            });
            // Run tests
            const testA = createMockTest('test-a', 'test a', '/project/tests/a.spec.ts');
            const testB = createMockTest('test-b', 'test b', '/project/tests/b.spec.ts');
            await reporter.onTestEnd(testA, createMockResult('passed', 100));
            await reporter.onTestEnd(testB, createMockResult('passed', 200));
            // End run
            await reporter.onEnd({ status: 'passed' });
            // Verify final events
            const runEventCalls = mockApiWriter.writeRunEvent.mock.calls;
            const lastRunEvent = runEventCalls[runEventCalls.length - 1][0];
            (0, vitest_1.expect)(lastRunEvent).toMatchObject({
                type: 'run.completed',
                status: 'passed',
                counts: { passed: 2, failed: 0, skipped: 0, flaky: 0 },
            });
            // Verify results were sent
            (0, vitest_1.expect)(mockApiWriter.writeResults).toHaveBeenCalled();
        });
    });
    (0, vitest_1.describe)('test run with failures', () => {
        (0, vitest_1.it)('records failed tests correctly', async () => {
            const reporter = new reporter_1.default({ mode: 'dev', flushThreshold: 100 });
            await reporter.onBegin(createMockConfig(), createMockSuite([]));
            const test = createMockTest('test-1', 'failing test', '/project/tests/fail.spec.ts');
            await reporter.onTestEnd(test, createMockResult('failed', 500, 0, { message: 'Expected true to be false' }));
            await reporter.onEnd({ status: 'failed' });
            const runEventCalls = mockApiWriter.writeRunEvent.mock.calls;
            const completedEvent = runEventCalls[runEventCalls.length - 1][0];
            (0, vitest_1.expect)(completedEvent.counts.failed).toBe(1);
            (0, vitest_1.expect)(completedEvent.status).toBe('failed');
        });
    });
    (0, vitest_1.describe)('test run with flaky tests', () => {
        (0, vitest_1.it)('records test as flaky when it fails then passes on retry', async () => {
            const reporter = new reporter_1.default({ mode: 'dev', flushThreshold: 100 });
            await reporter.onBegin(createMockConfig(), createMockSuite([]));
            const test = createMockTest('flaky-test', 'flaky test', '/project/tests/flaky.spec.ts');
            // First attempt fails
            await reporter.onTestEnd(test, createMockResult('failed', 500, 0, { message: 'Flaky failure' }));
            // Retry passes
            await reporter.onTestEnd(test, createMockResult('passed', 300, 1));
            await reporter.onEnd({ status: 'passed' });
            const runEventCalls = mockApiWriter.writeRunEvent.mock.calls;
            const completedEvent = runEventCalls[runEventCalls.length - 1][0];
            (0, vitest_1.expect)(completedEvent.counts.flaky).toBe(1);
            (0, vitest_1.expect)(completedEvent.counts.passed).toBe(0);
            (0, vitest_1.expect)(completedEvent.counts.failed).toBe(0);
        });
    });
    (0, vitest_1.describe)('test run with skipped tests', () => {
        (0, vitest_1.it)('records skipped tests correctly', async () => {
            const reporter = new reporter_1.default({ mode: 'dev', flushThreshold: 100 });
            await reporter.onBegin(createMockConfig(), createMockSuite([]));
            const test = createMockTest('skip-test', 'skipped test', '/project/tests/skip.spec.ts');
            await reporter.onTestEnd(test, createMockResult('skipped', 0));
            await reporter.onEnd({ status: 'passed' });
            const runEventCalls = mockApiWriter.writeRunEvent.mock.calls;
            const completedEvent = runEventCalls[runEventCalls.length - 1][0];
            (0, vitest_1.expect)(completedEvent.counts.skipped).toBe(1);
        });
    });
    (0, vitest_1.describe)('incremental flushing', () => {
        (0, vitest_1.it)('flushes results when threshold is reached', async () => {
            const reporter = new reporter_1.default({
                mode: 'dev',
                flushThreshold: 2,
                flushIntervalMs: 100000, // Long interval so only threshold triggers
            });
            await reporter.onBegin(createMockConfig(), createMockSuite([]));
            // Send 2 results to hit threshold
            await reporter.onTestEnd(createMockTest('t1', 'test 1', '/tests/1.spec.ts'), createMockResult('passed', 100));
            await reporter.onTestEnd(createMockTest('t2', 'test 2', '/tests/2.spec.ts'), createMockResult('passed', 100));
            // Should have flushed once (threshold = 2)
            (0, vitest_1.expect)(mockApiWriter.writeResults).toHaveBeenCalled();
            // Check batch ID is included
            const call = mockApiWriter.writeResults.mock.calls[0];
            (0, vitest_1.expect)(call[4]).toBe('001'); // batchId
        });
    });
    (0, vitest_1.describe)('mixed results scenario', () => {
        (0, vitest_1.it)('handles a realistic test run with mixed results', async () => {
            const reporter = new reporter_1.default({ mode: 'dev', flushThreshold: 100 });
            await reporter.onBegin(createMockConfig(), createMockSuite([]));
            // 3 passing tests
            for (let i = 1; i <= 3; i++) {
                await reporter.onTestEnd(createMockTest(`pass-${i}`, `passing test ${i}`, `/tests/pass${i}.spec.ts`), createMockResult('passed', 100 * i));
            }
            // 1 failing test
            await reporter.onTestEnd(createMockTest('fail-1', 'failing test', '/tests/fail.spec.ts'), createMockResult('failed', 500, 0, { message: 'Assertion error' }));
            // 1 flaky test (fail then pass)
            const flakyTest = createMockTest('flaky-1', 'flaky test', '/tests/flaky.spec.ts');
            await reporter.onTestEnd(flakyTest, createMockResult('failed', 200, 0));
            await reporter.onTestEnd(flakyTest, createMockResult('passed', 150, 1));
            // 1 skipped test
            await reporter.onTestEnd(createMockTest('skip-1', 'skipped test', '/tests/skip.spec.ts'), createMockResult('skipped', 0));
            await reporter.onEnd({ status: 'failed' });
            const runEventCalls = mockApiWriter.writeRunEvent.mock.calls;
            const completedEvent = runEventCalls[runEventCalls.length - 1][0];
            (0, vitest_1.expect)(completedEvent.counts).toEqual({
                passed: 3,
                failed: 1,
                skipped: 1,
                flaky: 1,
            });
        });
    });
    (0, vitest_1.describe)('timed out tests', () => {
        (0, vitest_1.it)('treats timedOut status as failed', async () => {
            const reporter = new reporter_1.default({ mode: 'dev', flushThreshold: 100 });
            await reporter.onBegin(createMockConfig(), createMockSuite([]));
            await reporter.onTestEnd(createMockTest('timeout-1', 'slow test', '/tests/slow.spec.ts'), createMockResult('timedOut', 30000));
            await reporter.onEnd({ status: 'timedOut' });
            const runEventCalls = mockApiWriter.writeRunEvent.mock.calls;
            const completedEvent = runEventCalls[runEventCalls.length - 1][0];
            (0, vitest_1.expect)(completedEvent.counts.failed).toBe(1);
        });
    });
});//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/reporter.test.js.map