"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
const path = __importStar(require("path"));
const fs = __importStar(require("fs"));
// Read version from package.json at runtime
const packageJsonPath = path.join(__dirname, '..', 'package.json');
const packageJson = JSON.parse(fs.readFileSync(packageJsonPath, 'utf-8'));
const VERSION = packageJson.version;
const types_1 = require("./types");
const utils_1 = require("./utils");
const buffers_1 = require("./buffers");
const counts_1 = require("./counts");
const payload_1 = require("./payload");
const api_1 = require("./storage/api");
class ParquetReporter {
    // Configuration
    mode;
    apiWriter = null;
    repoName;
    verbose;
    disabled = false;
    // Run metadata
    runId;
    startedAt = (0, utils_1.iso)();
    repoRoot;
    testDir;
    datePartition;
    defaultProjectName;
    projectRetries = new Map();
    // GitHub metadata
    branchName;
    sha;
    workflowName;
    jobName;
    runUrl;
    commitMessage;
    commitAuthor;
    runNumber;
    githubRunId;
    // Buffering & counting
    buffers = new Map();
    expectedTestsPerSpec = new Map();
    completedTestsPerSpec = new Map();
    countsState;
    // Run events
    runStartedEvent = null;
    // Options stored for onBegin
    shardName;
    // Incremental flush state
    flushIntervalMs = 30000; // Flush every 30 seconds
    flushThreshold = 10; // Flush after 10 results
    flushTimer = null;
    flushCounter = 0;
    pendingCount = 0; // Count of results pending flush (for threshold)
    constructor(options) {
        // Mode is required - get from options or env var
        const mode = options.mode || process.env.E2E_REPORTER_MODE;
        const validModes = ['dev', 'prod', 'disabled'];
        if (!mode) {
            console.error('[ParquetReporter] mode is required. Use "dev", "prod", or "disabled"');
            this.disabled = true;
            this.mode = 'disabled';
            this.repoName = '';
            this.runId = '';
            this.datePartition = '';
            this.repoRoot = '';
            this.verbose = false;
            this.countsState = {
                countedStatusPerTest: new Map(),
                totalPassed: 0,
                totalFailed: 0,
                totalSkipped: 0,
                totalFlaky: 0,
                buffers: this.buffers,
            };
            return;
        }
        if (!validModes.includes(mode)) {
            console.error(`[ParquetReporter] Invalid mode "${mode}". Use "dev", "prod", or "disabled"`);
            this.disabled = true;
            this.mode = 'disabled';
            this.repoName = '';
            this.runId = '';
            this.datePartition = '';
            this.repoRoot = '';
            this.verbose = false;
            this.countsState = {
                countedStatusPerTest: new Map(),
                totalPassed: 0,
                totalFailed: 0,
                totalSkipped: 0,
                totalFlaky: 0,
                buffers: this.buffers,
            };
            return;
        }
        this.mode = mode;
        // Handle disabled mode
        if (mode === 'disabled') {
            this.disabled = true;
            this.repoName = '';
            this.runId = '';
            this.datePartition = '';
            this.repoRoot = '';
            this.verbose = false;
            this.countsState = {
                countedStatusPerTest: new Map(),
                totalPassed: 0,
                totalFailed: 0,
                totalSkipped: 0,
                totalFlaky: 0,
                buffers: this.buffers,
            };
            return;
        }
        // Production mode requires CI environment (GitHub Actions)
        const isCI = process.env.GITHUB_ACTIONS === 'true' || process.env.CI === 'true';
        if (mode === 'prod' && !isCI) {
            console.log('✓ Reporter: skipped (prod requires CI)');
            this.disabled = true;
            this.repoName = '';
            this.runId = '';
            this.datePartition = '';
            this.repoRoot = '';
            this.verbose = false;
            this.countsState = {
                countedStatusPerTest: new Map(),
                totalPassed: 0,
                totalFailed: 0,
                totalSkipped: 0,
                totalFlaky: 0,
                buffers: this.buffers,
            };
            return;
        }
        // Identity
        this.repoName = (0, utils_1.getRepoName)(options.repoName);
        this.shardName = options.shardName; // Store for onBegin to use with config
        this.runId = (0, utils_1.getRunId)(this.shardName); // Preliminary runId, updated in onBegin with config
        this.datePartition = (0, utils_1.getDatePartition)();
        this.repoRoot = process.cwd();
        this.testDir = options.testDir || process.env.E2E_REPORTER_TEST_DIR;
        this.verbose = Boolean(options.verbose) || process.env.E2E_REPORTER_VERBOSE === '1';
        // Initialize API writer with hardcoded URL based on mode
        const apiUrl = types_1.API_URLS[mode];
        const apiKey = process.env.CONNECT_API_KEY;
        this.apiWriter = (0, api_1.createApiWriter)({ url: apiUrl, apiKey }, this.verbose);
        // Incremental flush configuration
        this.flushIntervalMs = options.flushIntervalMs ?? 30000;
        this.flushThreshold = options.flushThreshold ?? 10;
        // Initialize counts state
        this.countsState = {
            countedStatusPerTest: new Map(),
            totalPassed: 0,
            totalFailed: 0,
            totalSkipped: 0,
            totalFlaky: 0,
            buffers: this.buffers,
        };
        // Log reporter initialization
        console.log(`📡 Test Insights v${VERSION} ready`);
        if (this.verbose) {
            console.log(`   ${apiUrl} · ${this.runId} · ${this.datePartition}`);
        }
    }
    async onBegin(config, suite) {
        if (this.disabled)
            return;
        // Update runId with Playwright shard config for auto-detection
        // This enables auto-shard naming when using --shard flag without manual SHARD_NAME
        this.runId = (0, utils_1.getRunId)(this.shardName, config);
        // Extract GitHub metadata
        const ghMeta = (0, utils_1.getGithubMetadata)();
        this.branchName = ghMeta.branchName;
        this.sha = ghMeta.sha;
        this.workflowName = ghMeta.workflowName;
        this.jobName = ghMeta.jobName;
        this.runUrl = ghMeta.runUrl;
        this.runNumber = ghMeta.runNumber;
        this.githubRunId = ghMeta.githubRunId;
        // Fetch commit info from GitHub API (once per run)
        const commitInfo = await (0, utils_1.fetchCommitInfo)(this.sha);
        this.commitMessage = commitInfo.message;
        this.commitAuthor = commitInfo.author;
        this.startedAt = (0, utils_1.iso)();
        // Build run.started event
        const runCtx = {
            repoName: this.repoName,
            runId: this.runId,
            branchName: this.branchName,
            sha: this.sha,
            commitMessage: this.commitMessage,
            commitAuthor: this.commitAuthor,
        };
        this.runStartedEvent = (0, payload_1.buildRunStarted)(runCtx, {
            startedAt: this.startedAt,
            branch: this.branchName,
            sha: this.sha,
            workflowName: this.workflowName,
            jobName: this.jobName,
            runUrl: this.runUrl,
            commitMessage: this.commitMessage,
            commitAuthor: this.commitAuthor,
            runNumber: this.runNumber,
            githubRunId: this.githubRunId,
        });
        // Collect expected test counts
        try {
            this.collectExpectedTests(suite);
        }
        catch {
            /* ignore */
        }
        // Initialize project config
        this.initProjectConfig(config);
        // Write initial run.parquet with run.started event (for live visibility)
        await this.writeInitialRunFile();
        // Start periodic flush timer
        this.startFlushTimer();
        if (this.verbose) {
            const branch = this.branchName || '(local)';
            const run = this.runNumber ? `#${this.runNumber}` : '(GITHUB_RUN_NUMBER not set)';
            console.log(`  git: ${branch} | run: ${run} | specs: ${this.expectedTestsPerSpec.size}`);
            console.log(`  env: GITHUB_RUN_NUMBER=${process.env.GITHUB_RUN_NUMBER || '(unset)'}, GITHUB_RUN_ID=${process.env.GITHUB_RUN_ID || '(unset)'}`);
        }
    }
    async writeInitialRunFile() {
        if (!this.runStartedEvent || !this.apiWriter)
            return;
        // Write initial data.parquet with run metadata placeholder
        // This creates the file immediately so the run is visible in the dashboard
        // Even with no results, we need at least one row to create the file
        // So we create a placeholder row with run metadata only
        const placeholderRow = {
            repoName: this.repoName,
            runId: this.runId,
            specPath: '__run_started__',
            testName: '__run_started__',
            status: 'passed',
            eventId: `${this.runId}::__run_started__`,
            timestamp: this.startedAt,
            runStartedAt: this.startedAt,
            workflowName: this.workflowName,
            jobName: this.jobName,
            runUrl: this.runUrl,
            runNumber: this.runNumber,
            githubRunId: this.githubRunId,
            branch: this.branchName,
            sha: this.sha,
            commitMessage: this.commitMessage,
            commitAuthor: this.commitAuthor,
        };
        await this.apiWriter.writeData([placeholderRow], this.repoName, this.datePartition, this.runId);
    }
    startFlushTimer() {
        if (this.flushTimer)
            return;
        this.flushTimer = setInterval(async () => {
            await this.flushPendingResults();
        }, this.flushIntervalMs);
    }
    stopFlushTimer() {
        if (this.flushTimer) {
            clearInterval(this.flushTimer);
            this.flushTimer = null;
        }
    }
    async flushPendingResults() {
        if (this.pendingCount === 0 || !this.apiWriter)
            return;
        // Get all results from buffers (single source of truth)
        const allResults = (0, buffers_1.allValues)(this.buffers);
        if (allResults.length === 0)
            return;
        const newResultsCount = this.pendingCount;
        this.pendingCount = 0;
        this.flushCounter++;
        // Write ALL results so far (overwrites data.parquet)
        const consolidatedData = this.buildConsolidatedData(allResults);
        const success = await this.apiWriter.writeData(consolidatedData, this.repoName, this.datePartition, this.runId);
        if (success) {
            if (this.verbose) {
                console.log(`  ↑ flushed ${allResults.length} total results (+${newResultsCount} new)`);
            }
        }
        else {
            // On API failure, restore pending count for retry
            this.pendingCount = newResultsCount;
            this.flushCounter--;
        }
    }
    async onTestEnd(test, result) {
        if (this.disabled)
            return;
        const file = test.location?.file || 'unknown';
        const spec = (0, utils_1.toRelativeSpec)(this.repoRoot, file);
        const { project, status, attempt, browser, osName, titlePath } = this.analyzeTestOutcome(test, result);
        const runCtx = {
            repoName: this.repoName,
            runId: this.runId,
            branchName: this.branchName,
            sha: this.sha,
            commitMessage: this.commitMessage,
            commitAuthor: this.commitAuthor,
        };
        const payload = (0, payload_1.buildResultPayload)(runCtx, spec, project, browser, osName, test, result, status, attempt, titlePath);
        // Use stable test.id for the key
        const testKey = `${spec}::${test.id}`;
        // Buffer the result (latest attempt overwrites previous)
        (0, buffers_1.setTest)(this.buffers, spec, testKey, payload);
        // Only count as pending if this is a FINAL result
        // (passed, skipped, or failed on last retry attempt)
        // This prevents intermediate retry failures from triggering flush
        const isFinalResult = result.status === 'passed' ||
            result.status === 'skipped' ||
            result.retry >= (test.retries || 0);
        if (isFinalResult) {
            this.pendingCount++;
        }
        // Track completion
        const completed = (this.completedTestsPerSpec.get(spec) || 0) + 1;
        this.completedTestsPerSpec.set(spec, completed);
        if (this.verbose) {
            const statusIcon = status === 'passed'
                ? '✓'
                : status === 'failed'
                    ? '✗'
                    : status === 'skipped'
                        ? '○'
                        : '⟳';
            const attemptStr = attempt > 1 ? ` (attempt ${attempt})` : '';
            console.log(`  ${statusIcon} ${test.title}${attemptStr}`);
        }
        // Flush if threshold reached
        if (this.pendingCount >= this.flushThreshold) {
            await this.flushPendingResults();
        }
    }
    async onEnd(result) {
        if (this.disabled)
            return;
        // Stop the periodic flush timer
        this.stopFlushTimer();
        try {
            // Flush any remaining pending results
            if (this.pendingCount > 0) {
                await this.flushPendingResults();
            }
            // Calculate final counts from all buffered results
            // Use testId (not testName) so repeat-each iterations are counted separately
            const allResults = (0, buffers_1.allValues)(this.buffers);
            for (const r of allResults) {
                const testKey = `${r.runId}::${r.specPath}::${r.testId}`;
                (0, counts_1.updateRunTotalsForTest)(this.countsState, testKey, r.status);
            }
            const counts = (0, counts_1.getFinalCounts)(this.countsState);
            const durationMs = Date.now() - new Date(this.startedAt).getTime();
            // Build run.completed event
            const runCtx = {
                repoName: this.repoName,
                runId: this.runId,
                branchName: this.branchName,
                sha: this.sha,
                commitMessage: this.commitMessage,
                commitAuthor: this.commitAuthor,
            };
            const runCompletedEvent = (0, payload_1.buildRunCompleted)(runCtx, {
                completedAt: (0, utils_1.iso)(),
                status: result.status,
                durationMs,
                counts,
            });
            // Write run.parquet and consolidated results.parquet
            const uploadSuccess = await this.writeFinalFiles(runCompletedEvent, allResults);
            // Always show summary (even in non-verbose mode)
            console.log('==================================');
            console.log(`✔ ${counts.passed} passed   ✖ ${counts.failed} failed   ⚠ ${counts.flaky} flaky   ➖ ${counts.skipped} skipped`);
            if (uploadSuccess) {
                console.log(`📡 All results recorded (${allResults.length} tests)`);
            }
            else {
                console.log(`📡 Failed to record results`);
            }
            console.log('==================================');
        }
        catch (err) {
            console.error('[ParquetReporter] onEnd error:', err.message);
            if (this.verbose) {
                console.error(err);
            }
        }
    }
    async writeFinalFiles(runCompleted, allResults) {
        if (!this.apiWriter)
            return false;
        try {
            // Write consolidated data (single file with embedded run metadata)
            const consolidatedData = this.buildConsolidatedData(allResults, {
                runCompletedAt: runCompleted.completedAt,
                runStatus: runCompleted.status,
                runDurationMs: runCompleted.durationMs,
                countsPassed: runCompleted.counts.passed,
                countsFailed: runCompleted.counts.failed,
                countsSkipped: runCompleted.counts.skipped,
                countsFlaky: runCompleted.counts.flaky,
            });
            await this.apiWriter.writeData(consolidatedData, this.repoName, this.datePartition, this.runId);
            if (this.verbose) {
                console.log(`  ↑ sent ${consolidatedData.length} consolidated results to API`);
            }
            return true;
        }
        catch {
            return false;
        }
    }
    /**
     * Convert results to consolidated data format with embedded run metadata
     */
    buildConsolidatedData(results, runCompletion) {
        return results.map((r) => ({
            // Test result fields
            repoName: r.repoName,
            runId: r.runId,
            specPath: r.specPath,
            project: r.project,
            testName: r.testName,
            testId: r.testId,
            status: r.status,
            durationMs: r.durationMs,
            startedAt: r.startedAt,
            attempt: r.attempt,
            retries: r.retries,
            failureMessage: r.failureMessage,
            eventId: r.eventId,
            timestamp: r.timestamp,
            browser: r.browser,
            os: r.os,
            branch: r.branch,
            sha: r.sha,
            reportUrl: r.reportUrl,
            commitMessage: r.commitMessage,
            commitAuthor: r.commitAuthor,
            titlePath: r.titlePath,
            // Run metadata (embedded)
            runStartedAt: this.startedAt,
            runCompletedAt: runCompletion?.runCompletedAt,
            runStatus: runCompletion?.runStatus,
            runDurationMs: runCompletion?.runDurationMs,
            workflowName: this.workflowName,
            jobName: this.jobName,
            runUrl: this.runUrl,
            runNumber: this.runNumber,
            githubRunId: this.githubRunId,
            // Counts (only populated at completion)
            countsPassed: runCompletion?.countsPassed,
            countsFailed: runCompletion?.countsFailed,
            countsSkipped: runCompletion?.countsSkipped,
            countsFlaky: runCompletion?.countsFlaky,
        }));
    }
    initProjectConfig(config) {
        try {
            const cfg = config;
            const firstProj = Array.isArray(cfg.projects) ? cfg.projects[0] : undefined;
            if (!this.testDir)
                this.testDir = cfg.testDir || firstProj?.testDir;
            if (!this.defaultProjectName)
                this.defaultProjectName = firstProj?.name;
            if (this.testDir && !path.isAbsolute(this.testDir)) {
                this.testDir = path.resolve(this.repoRoot, this.testDir);
            }
        }
        catch {
            /* ignore */
        }
        try {
            const cfg = config;
            const globalRetries = typeof cfg.retries === 'number' ? cfg.retries : 0;
            if (Array.isArray(cfg.projects)) {
                for (const p of cfg.projects) {
                    const name = p?.name ?? 'default';
                    const pr = typeof p?.retries === 'number' ? p.retries : globalRetries;
                    this.projectRetries.set(name, pr);
                }
            }
            else {
                this.projectRetries.set(this.defaultProjectName || 'default', globalRetries);
            }
        }
        catch {
            /* ignore */
        }
    }
    analyzeTestOutcome(test, result) {
        // Extract project name and title path
        let titlePathArr;
        const tpUnknown = test.titlePath;
        if (typeof tpUnknown === 'function') {
            try {
                titlePathArr = tpUnknown.call(test);
            }
            catch {
                /* ignore */
            }
        }
        else if (Array.isArray(tpUnknown)) {
            titlePathArr = tpUnknown;
        }
        const project = titlePathArr?.[0] || this.defaultProjectName;
        // Status mapping
        const baseStatus = result.status === 'timedOut' ? 'failed' : result.status;
        const attempt = typeof result.retry === 'number' ? result.retry + 1 : 1;
        const status = attempt > 1 && baseStatus === 'passed' ? 'flaky' : baseStatus;
        const browser = (0, utils_1.inferBrowserFromProject)(project);
        const osName = (0, utils_1.inferOs)();
        // Return titlePath for test hierarchy (excluding project name at index 0)
        const titlePath = titlePathArr?.slice(1);
        return { project, status, attempt, browser, osName, titlePath };
    }
    collectExpectedTests(node) {
        if (node == null || typeof node !== 'object')
            return;
        const rec = node;
        const type = typeof rec.type === 'string' ? rec.type : undefined;
        const location = rec.location && typeof rec.location === 'object'
            ? rec.location
            : undefined;
        if (type === 'test') {
            const file = location?.file || 'unknown';
            const rel = (0, utils_1.toRelativeSpec)(this.repoRoot, file);
            this.expectedTestsPerSpec.set(rel, (this.expectedTestsPerSpec.get(rel) || 0) + 1);
            return;
        }
        const gather = (key) => {
            const v = rec[key];
            return Array.isArray(v) ? v : [];
        };
        const children = [
            ...gather('suites'),
            ...gather('tests'),
            ...gather('children'),
            ...gather('entries'),
        ];
        for (const c of children)
            this.collectExpectedTests(c);
    }
}
exports.default = ParquetReporter;//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/reporter.js.map