"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vitest_1 = require("vitest");
const payload_1 = require("./payload");
// Mock the utils module
vitest_1.vi.mock('./utils', () => ({
    iso: (date) => date?.toISOString() ?? '2024-01-15T10:30:00.000Z',
    b64url: (str) => Buffer.from(str).toString('base64url'),
    buildReportUrl: (testId) => `https://example.com/report/${testId}`,
}));
(0, vitest_1.describe)('buildResultPayload', () => {
    const runCtx = {
        repoName: 'test-repo',
        runId: 'run-123',
        branchName: 'main',
        sha: 'abc123def',
        commitMessage: 'fix: something',
        commitAuthor: 'dev@example.com',
    };
    const mockTest = {
        id: 'test-id-123',
        title: 'should work correctly',
        location: { file: '/path/to/test.spec.ts', line: 10, column: 5 },
    };
    const mockResult = {
        status: 'passed',
        duration: 1500,
        startTime: new Date('2024-01-15T10:30:00.000Z'),
        retry: 0,
        error: undefined,
    };
    (0, vitest_1.it)('builds valid result payload with all fields', () => {
        const payload = (0, payload_1.buildResultPayload)(runCtx, 'tests/example.spec.ts', 'chromium', 'chromium', 'mac', mockTest, mockResult, 'passed', 1, ['Suite', 'Nested']);
        (0, vitest_1.expect)(payload.type).toBe('result.created');
        (0, vitest_1.expect)(payload.repoName).toBe('test-repo');
        (0, vitest_1.expect)(payload.runId).toBe('run-123');
        (0, vitest_1.expect)(payload.specPath).toBe('tests/example.spec.ts');
        (0, vitest_1.expect)(payload.project).toBe('chromium');
        (0, vitest_1.expect)(payload.browser).toBe('chromium');
        (0, vitest_1.expect)(payload.os).toBe('mac');
        (0, vitest_1.expect)(payload.testName).toBe('should work correctly');
        (0, vitest_1.expect)(payload.testId).toBe('test-id-123');
        (0, vitest_1.expect)(payload.status).toBe('passed');
        (0, vitest_1.expect)(payload.durationMs).toBe(1500);
        (0, vitest_1.expect)(payload.attempt).toBe(1);
        (0, vitest_1.expect)(payload.branch).toBe('main');
        (0, vitest_1.expect)(payload.sha).toBe('abc123def');
        (0, vitest_1.expect)(payload.commitMessage).toBe('fix: something');
        (0, vitest_1.expect)(payload.commitAuthor).toBe('dev@example.com');
        (0, vitest_1.expect)(payload.titlePath).toEqual(['Suite', 'Nested']);
        (0, vitest_1.expect)(payload.eventId).toBeDefined();
        (0, vitest_1.expect)(payload.timestamp).toBeDefined();
    });
    (0, vitest_1.it)('builds valid payload with minimal context', () => {
        const minimalCtx = {
            repoName: 'test-repo',
            runId: 'run-123',
        };
        const payload = (0, payload_1.buildResultPayload)(minimalCtx, 'tests/example.spec.ts', undefined, undefined, 'linux', mockTest, mockResult, 'passed', 1);
        (0, vitest_1.expect)(payload.type).toBe('result.created');
        (0, vitest_1.expect)(payload.repoName).toBe('test-repo');
        (0, vitest_1.expect)(payload.project).toBeUndefined();
        (0, vitest_1.expect)(payload.browser).toBeUndefined();
        (0, vitest_1.expect)(payload.branch).toBeUndefined();
    });
    (0, vitest_1.it)('handles flaky status for retry passes', () => {
        const payload = (0, payload_1.buildResultPayload)(runCtx, 'tests/example.spec.ts', 'chromium', 'chromium', 'mac', mockTest, mockResult, 'flaky', // status already determined as flaky
        2 // second attempt
        );
        (0, vitest_1.expect)(payload.status).toBe('flaky');
        (0, vitest_1.expect)(payload.attempt).toBe(2);
    });
    (0, vitest_1.it)('includes failure message when present', () => {
        const failedResult = {
            ...mockResult,
            status: 'failed',
            error: { message: 'Expected true but got false' },
        };
        const payload = (0, payload_1.buildResultPayload)(runCtx, 'tests/example.spec.ts', 'chromium', 'chromium', 'mac', mockTest, failedResult, 'failed', 1);
        (0, vitest_1.expect)(payload.status).toBe('failed');
        (0, vitest_1.expect)(payload.failureMessage).toBe('Expected true but got false');
    });
    (0, vitest_1.it)('generates idempotent eventId without attempt number', () => {
        const payload1 = (0, payload_1.buildResultPayload)(runCtx, 'tests/example.spec.ts', 'chromium', 'chromium', 'mac', mockTest, mockResult, 'failed', 1);
        const payload2 = (0, payload_1.buildResultPayload)(runCtx, 'tests/example.spec.ts', 'chromium', 'chromium', 'mac', mockTest, mockResult, 'passed', 2);
        // Same test should have same eventId regardless of attempt
        (0, vitest_1.expect)(payload1.eventId).toBe(payload2.eventId);
    });
    (0, vitest_1.it)('throws on invalid payload', () => {
        const badCtx = {
            repoName: '', // empty string should fail validation
            runId: 'run-123',
        };
        (0, vitest_1.expect)(() => (0, payload_1.buildResultPayload)(badCtx, 'tests/example.spec.ts', undefined, undefined, 'mac', mockTest, mockResult, 'passed', 1)).toThrow();
    });
});
(0, vitest_1.describe)('buildRunStarted', () => {
    const runCtx = {
        repoName: 'test-repo',
        runId: 'run-123',
    };
    (0, vitest_1.it)('builds valid run started event', () => {
        const event = (0, payload_1.buildRunStarted)(runCtx, {
            startedAt: '2024-01-15T10:30:00.000Z',
            branch: 'main',
            sha: 'abc123',
            workflowName: 'CI',
            jobName: 'test',
            runUrl: 'https://github.com/org/repo/actions/runs/123',
            commitMessage: 'fix: bug',
            commitAuthor: 'dev@example.com',
        });
        (0, vitest_1.expect)(event.type).toBe('run.started');
        (0, vitest_1.expect)(event.repoName).toBe('test-repo');
        (0, vitest_1.expect)(event.runId).toBe('run-123');
        (0, vitest_1.expect)(event.startedAt).toBe('2024-01-15T10:30:00.000Z');
        (0, vitest_1.expect)(event.branch).toBe('main');
        (0, vitest_1.expect)(event.sha).toBe('abc123');
        (0, vitest_1.expect)(event.workflowName).toBe('CI');
        (0, vitest_1.expect)(event.jobName).toBe('test');
        (0, vitest_1.expect)(event.runUrl).toBe('https://github.com/org/repo/actions/runs/123');
    });
    (0, vitest_1.it)('builds valid event with minimal options', () => {
        const event = (0, payload_1.buildRunStarted)(runCtx, {
            startedAt: '2024-01-15T10:30:00.000Z',
        });
        (0, vitest_1.expect)(event.type).toBe('run.started');
        (0, vitest_1.expect)(event.startedAt).toBe('2024-01-15T10:30:00.000Z');
        (0, vitest_1.expect)(event.branch).toBeUndefined();
    });
    (0, vitest_1.it)('throws on invalid event', () => {
        const badCtx = {
            repoName: '',
            runId: 'run-123',
        };
        (0, vitest_1.expect)(() => (0, payload_1.buildRunStarted)(badCtx, {
            startedAt: '2024-01-15T10:30:00.000Z',
        })).toThrow();
    });
});
(0, vitest_1.describe)('buildRunCompleted', () => {
    const runCtx = {
        repoName: 'test-repo',
        runId: 'run-123',
    };
    (0, vitest_1.it)('builds valid run completed event', () => {
        const event = (0, payload_1.buildRunCompleted)(runCtx, {
            completedAt: '2024-01-15T10:35:00.000Z',
            status: 'passed',
            durationMs: 300000,
            counts: {
                passed: 10,
                failed: 0,
                skipped: 2,
                flaky: 1,
            },
        });
        (0, vitest_1.expect)(event.type).toBe('run.completed');
        (0, vitest_1.expect)(event.repoName).toBe('test-repo');
        (0, vitest_1.expect)(event.runId).toBe('run-123');
        (0, vitest_1.expect)(event.completedAt).toBe('2024-01-15T10:35:00.000Z');
        (0, vitest_1.expect)(event.status).toBe('passed');
        (0, vitest_1.expect)(event.durationMs).toBe(300000);
        (0, vitest_1.expect)(event.counts.passed).toBe(10);
        (0, vitest_1.expect)(event.counts.failed).toBe(0);
        (0, vitest_1.expect)(event.counts.skipped).toBe(2);
        (0, vitest_1.expect)(event.counts.flaky).toBe(1);
    });
    (0, vitest_1.it)('builds valid event with failed status', () => {
        const event = (0, payload_1.buildRunCompleted)(runCtx, {
            completedAt: '2024-01-15T10:35:00.000Z',
            status: 'failed',
            durationMs: 300000,
            counts: {
                passed: 8,
                failed: 2,
                skipped: 0,
                flaky: 0,
            },
        });
        (0, vitest_1.expect)(event.status).toBe('failed');
        (0, vitest_1.expect)(event.counts.failed).toBe(2);
    });
    (0, vitest_1.it)('throws on invalid counts', () => {
        (0, vitest_1.expect)(() => (0, payload_1.buildRunCompleted)(runCtx, {
            completedAt: '2024-01-15T10:35:00.000Z',
            status: 'passed',
            durationMs: 300000,
            counts: {
                passed: -1, // invalid
                failed: 0,
                skipped: 0,
                flaky: 0,
            },
        })).toThrow();
    });
});//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/payload.test.js.map