"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildResultPayload = buildResultPayload;
exports.buildRunStarted = buildRunStarted;
exports.buildRunCompleted = buildRunCompleted;
const types_1 = require("./types");
const utils_1 = require("./utils");
function buildResultPayload(runCtx, spec, project, browser, osName, test, result, status, attempt, titlePath) {
    const { repoName, runId, branchName, sha, commitMessage, commitAuthor } = runCtx;
    // Use Playwright's stable test.id for consistent identification across runs
    const stableTestId = test.id;
    // Idempotent event id without attempt to allow later attempts to replace
    const eventKey = `${runId}::${spec}::${stableTestId}`;
    const event = {
        type: 'result.created',
        repoName,
        runId,
        specPath: spec,
        project,
        browser,
        os: osName,
        branch: branchName,
        sha,
        testName: test.title,
        testId: stableTestId,
        status,
        durationMs: result.duration,
        startedAt: result.startTime ? (0, utils_1.iso)(result.startTime) : undefined,
        attempt,
        retries: typeof result.retry === 'number' ? result.retry : undefined,
        failureMessage: result.error?.message || undefined,
        eventId: (0, utils_1.b64url)(eventKey),
        timestamp: (0, utils_1.iso)(),
        reportUrl: (0, utils_1.buildReportUrl)(stableTestId),
        commitMessage,
        commitAuthor,
        titlePath,
    };
    // Validate event against schema - throw if invalid to prevent bad data
    const validation = (0, types_1.safeValidateResultEvent)(event);
    if (!validation.success) {
        console.error('[ParquetReporter] Invalid result event:', validation.error.message);
        throw new Error(`Invalid result event: ${validation.error.message}`);
    }
    return validation.data;
}
function buildRunStarted(runCtx, opts) {
    const event = {
        type: 'run.started',
        repoName: runCtx.repoName,
        runId: runCtx.runId,
        startedAt: opts.startedAt,
        branch: opts.branch,
        sha: opts.sha,
        workflowName: opts.workflowName,
        jobName: opts.jobName,
        runUrl: opts.runUrl,
        commitMessage: opts.commitMessage,
        commitAuthor: opts.commitAuthor,
        runNumber: opts.runNumber,
        githubRunId: opts.githubRunId,
    };
    // Validate event against schema - throw if invalid to prevent bad data
    const validation = (0, types_1.safeValidateRunStartedEvent)(event);
    if (!validation.success) {
        console.error('[ParquetReporter] Invalid RunStarted event:', validation.error.message);
        throw new Error(`Invalid RunStarted event: ${validation.error.message}`);
    }
    return validation.data;
}
function buildRunCompleted(runCtx, opts) {
    const event = {
        type: 'run.completed',
        repoName: runCtx.repoName,
        runId: runCtx.runId,
        completedAt: opts.completedAt,
        status: opts.status,
        durationMs: opts.durationMs,
        counts: opts.counts,
    };
    // Validate event against schema - throw if invalid to prevent bad data
    const validation = (0, types_1.safeValidateRunCompletedEvent)(event);
    if (!validation.success) {
        console.error('[ParquetReporter] Invalid RunCompleted event:', validation.error.message);
        throw new Error(`Invalid RunCompleted event: ${validation.error.message}`);
    }
    return validation.data;
}//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/693b6d13ba5d61566bec7f5a4a46126eff7bbbe1/node_modules/@midleman/playwright-reporter/dist/payload.js.map